
'use client';

import { useState, useMemo, useEffect } from 'react';
import { useQuizStore } from '@/hooks/use-quiz-store';
import { Card, CardHeader, CardTitle, CardContent, CardFooter, CardDescription } from '../ui/card';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { Checkbox } from '../ui/checkbox';
import { ScrollArea } from '../ui/scroll-area';
import { Shuffle } from 'lucide-react';
import { RadioGroup, RadioGroupItem } from '../ui/radio-group';

interface RandomQuizFormProps {
  setSelectedCategory: (category: string | null) => void;
}

export default function RandomQuizForm({ setSelectedCategory }: RandomQuizFormProps) {
  const { questions, startRandomQuiz } = useQuizStore();
  const [numQuestions, setNumQuestions] = useState(10);
  const [selectedCategories, setSelectedCategories] = useState<string[]>([]);
  const [isAllSelected, setIsAllSelected] = useState(false);
  const [quizType, setQuizType] = useState<'practice' | 'timed'>('practice');
  const [timeLimit, setTimeLimit] = useState(120);

  const uniqueCategories = useMemo(() => {
    return [...new Set(questions.map(q => q.category))];
  }, [questions]);
  
  useEffect(() => {
    if (selectedCategories.length === uniqueCategories.length && uniqueCategories.length > 0) {
        setIsAllSelected(true);
    } else {
        setIsAllSelected(false);
    }
  }, [selectedCategories, uniqueCategories]);

  const handleCategoryChange = (category: string) => {
    setSelectedCategories(prev =>
      prev.includes(category)
        ? prev.filter(c => c !== category)
        : [...prev, category]
    );
  };
  
  const handleSelectAllChange = () => {
    if (isAllSelected) {
        setSelectedCategories([]);
    } else {
        setSelectedCategories(uniqueCategories);
    }
    setIsAllSelected(!isAllSelected);
  }

  const handleStartRandomQuiz = () => {
    if (selectedCategories.length === 0) {
        alert("Please select at least one category.");
        return;
    }
    startRandomQuiz(numQuestions, selectedCategories, quizType, timeLimit);
    setSelectedCategory('Random Quiz');
  };

  return (
    <Card className="shadow-lg">
      <CardHeader>
        <div className="flex items-center gap-4">
            <div className="bg-primary/10 rounded-lg p-3 w-fit h-fit">
                <Shuffle className="h-8 w-8 text-primary" />
            </div>
            <div>
                <CardTitle className="font-headline text-2xl">Create a Random Quiz</CardTitle>
                <CardDescription>Mix questions from your favorite categories.</CardDescription>
            </div>
        </div>
      </CardHeader>
      <CardContent className="space-y-6">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-2">
                <Label>Quiz Type</Label>
                <RadioGroup defaultValue="practice" onValueChange={(value: 'practice' | 'timed') => setQuizType(value)}>
                    <div className="flex items-center space-x-2">
                        <RadioGroupItem value="practice" id="practice" />
                        <Label htmlFor="practice" className="font-normal cursor-pointer">Practice</Label>
                    </div>
                    <div className="flex items-center space-x-2">
                        <RadioGroupItem value="timed" id="timed" />
                        <Label htmlFor="timed" className="font-normal cursor-pointer">Time-limited Test</Label>
                    </div>
                </RadioGroup>
            </div>
            {quizType === 'timed' && (
                <div className="space-y-2">
                    <Label htmlFor="time-limit">Time Limit (minutes)</Label>
                    <Input
                        id="time-limit"
                        type="number"
                        value={timeLimit}
                        onChange={(e) => setTimeLimit(Number(e.target.value))}
                        min="1"
                    />
                </div>
            )}
        </div>
        <div className="space-y-2">
          <Label htmlFor="num-questions">Number of Questions</Label>
          <Input
            id="num-questions"
            type="number"
            value={numQuestions}
            onChange={(e) => setNumQuestions(Number(e.target.value))}
            min="1"
            max={questions.length}
          />
        </div>
        <div className="space-y-2">
          <Label>Categories</Label>
          <div className="flex items-center space-x-2 mb-2">
              <Checkbox
                id="all-categories"
                checked={isAllSelected}
                onCheckedChange={handleSelectAllChange}
              />
              <Label htmlFor="all-categories" className="font-bold cursor-pointer">All Categories</Label>
          </div>
          <ScrollArea className="h-40 w-full rounded-md border p-4">
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
              {uniqueCategories.map(category => (
                <div key={category} className="flex items-center space-x-2">
                  <Checkbox
                    id={category}
                    checked={selectedCategories.includes(category)}
                    onCheckedChange={() => handleCategoryChange(category)}
                  />
                  <Label htmlFor={category} className="font-normal cursor-pointer">{category}</Label>
                </div>
              ))}
            </div>
          </ScrollArea>
        </div>
      </CardContent>
      <CardFooter>
        <Button className="w-full" onClick={handleStartRandomQuiz}>
          Start Random Quiz
        </Button>
      </CardFooter>
    </Card>
  );
}
