
'use client';

import { useState, useMemo, useEffect } from 'react';
import { useQuizStore } from '@/hooks/use-quiz-store';
import QuizClient from './QuizClient';
import { Card, CardContent, CardFooter, CardHeader, CardTitle } from '../ui/card';
import { Button } from '../ui/button';
import { BrainCircuit, BookCopy, ChevronLeft } from 'lucide-react';
import { Badge } from '../ui/badge';
import RandomQuizForm from './RandomQuizForm';
import { Separator } from '../ui/separator';

export default function QuizLobby() {
  const { questions, isInitialized, startNewQuizAttempt, resetStats, resetUserAnswers, activeQuestions, quizType } = useQuizStore();
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);

  // Effect to reset stats when returning to the lobby
  useEffect(() => {
    if (!selectedCategory) {
      // We are in the lobby, so reset stats for the next attempt
      resetStats();
      resetUserAnswers();
       // Clear quiz progress from local storage when returning to lobby
       questions.forEach(q => {
        try {
          const storageKey = `medquiz-progress-${q.category}`;
          localStorage.removeItem(storageKey);
          if (q.category !== 'All Questions') {
              const allCatKey = `medquiz-progress-All Questions`;
              localStorage.removeItem(allCatKey);
          }
        } catch (error) {
            console.error("Failed to remove quiz progress from localStorage", error);
        }
       });
       localStorage.removeItem('medquiz-progress-Random Quiz');
    }
  }, [selectedCategory, resetStats, resetUserAnswers, questions]);

  const categories = useMemo(() => {
    if (!questions) return [];
    const uniqueCategories = [...new Set(questions.map(q => q.category))];
    return ['All Questions', ...uniqueCategories];
  }, [questions]);
  
  const handleStartQuiz = (category: string) => {
    startNewQuizAttempt(category);
    setSelectedCategory(category);
  };
  
  const handleBackToCategories = () => {
      setSelectedCategory(null);
  }

  if (!isInitialized) {
    return <div className="flex justify-center items-center h-64"><BrainCircuit className="h-12 w-12 animate-pulse text-primary" /></div>;
  }
  
  if (selectedCategory) {
    return (
        <div>
            <div className="mb-4">
              <Button onClick={handleBackToCategories} variant="outline">
                <ChevronLeft className="mr-2 h-4 w-4" />
                Back to Categories
              </Button>
            </div>
            <QuizClient questions={activeQuestions} category={selectedCategory} quizType={quizType} />
        </div>
    );
  }

  return (
    <>
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {categories.map(category => {
              const count = category === 'All Questions' ? questions.length : questions.filter(q => q.category === category).length;
              return (
                  <Card key={category} className="flex flex-col">
                      <CardHeader>
                          <div className="flex items-center gap-4">
                               <div className="bg-secondary p-3 rounded-lg">
                                  <BookCopy className="h-6 w-6 text-secondary-foreground" />
                              </div>
                              <CardTitle className="font-headline text-xl">{category}</CardTitle>
                          </div>
                      </CardHeader>
                      <CardContent className="flex-grow">
                         <Badge variant="outline">{count} Questions</Badge>
                      </CardContent>
                      <CardFooter>
                          <Button className="w-full" onClick={() => handleStartQuiz(category)} disabled={count === 0}>
                              Start Quiz
                          </Button>
                      </CardFooter>
                  </Card>
              )
          })}
      </div>
      <Separator className="my-8" />
      <RandomQuizForm setSelectedCategory={setSelectedCategory} />
    </>
  );
}
