
'use client';

import { useState } from 'react';
import { useQuizStore } from '@/hooks/use-quiz-store';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogFooter,
  DialogClose,
} from '@/components/ui/dialog';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog"

import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { PlusCircle, Edit, Trash2, BrainCircuit } from 'lucide-react';
import { useForm, useFieldArray } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import * as z from 'zod';
import type { Question } from '@/lib/types';
import { useToast } from '@/hooks/use-toast';
import { RadioGroup, RadioGroupItem } from '../ui/radio-group';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '../ui/form';
import { Textarea } from '../ui/textarea';

const questionSchema = z.object({
  questionText: z.string().min(10, 'Question must be at least 10 characters long.'),
  category: z.string().min(3, 'Category must be at least 3 characters long.'),
  options: z.array(z.string().min(1, "Option can't be empty.")).length(4, 'There must be exactly 4 options.'),
  correctAnswer: z.string().min(1, 'You must select a correct answer.'),
  explanation: z.string().min(10, 'Explanation must be at least 10 characters long.'),
});

type QuestionFormData = z.infer<typeof questionSchema>;

const QuestionForm = ({
  question,
  onSave,
  onClose,
}: {
  question?: Question | null;
  onSave: (data: QuestionFormData) => void;
  onClose: () => void;
}) => {
  const form = useForm<QuestionFormData>({
    resolver: zodResolver(questionSchema),
    defaultValues: {
      questionText: question?.questionText ?? '',
      category: question?.category ?? '',
      options: question?.options ?? ['', '', '', ''],
      correctAnswer: question?.correctAnswer ?? '',
      explanation: question?.explanation ?? '',
    },
  });

  const { fields } = useFieldArray({ control: form.control, name: 'options' });

  const watchedOptions = form.watch('options');

  const onSubmit = (data: QuestionFormData) => {
    onSave(data);
    onClose();
  };

  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <FormField
          control={form.control}
          name="questionText"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Question</FormLabel>
              <FormControl>
                <Input placeholder="What is the capital of..." {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        <FormField
          control={form.control}
          name="category"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Category</FormLabel>
              <FormControl>
                <Input placeholder="e.g. Cardiology" {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        
        <div className="space-y-2">
            <Label>Options</Label>
            {fields.map((field, index) => (
                <FormField
                    key={field.id}
                    control={form.control}
                    name={`options.${index}`}
                    render={({ field }) => (
                        <FormItem>
                        <FormControl>
                            <Input placeholder={`Option ${index + 1}`} {...field} />
                        </FormControl>
                        <FormMessage />
                        </FormItem>
                    )}
                />
            ))}
        </div>
        
        <FormField
          control={form.control}
          name="correctAnswer"
          render={({ field }) => (
            <FormItem className="space-y-3">
              <FormLabel>Correct Answer</FormLabel>
              <FormControl>
                <RadioGroup
                  onValueChange={field.onChange}
                  defaultValue={field.value}
                  className="flex flex-col space-y-1"
                >
                  {watchedOptions.map((option, index) => (
                    <FormItem key={index} className="flex items-center space-x-3 space-y-0">
                      <FormControl>
                        <RadioGroupItem value={option} disabled={!option} />
                      </FormControl>
                      <FormLabel className="font-normal">{option || `Option ${index + 1}`}</FormLabel>
                    </FormItem>
                  ))}
                </RadioGroup>
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />
        
        <FormField
          control={form.control}
          name="explanation"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Explanation</FormLabel>
              <FormControl>
                <Textarea placeholder="Provide a brief explanation for the correct answer..." {...field} />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />


        <DialogFooter>
          <DialogClose asChild>
            <Button type="button" variant="outline">Cancel</Button>
          </DialogClose>
          <Button type="submit">Save Question</Button>
        </DialogFooter>
      </form>
    </Form>
  );
};


export default function ManageQuestionsClient() {
  const { questions, addQuestion, updateQuestion, removeQuestion, isInitialized } = useQuizStore();
  const [isFormOpen, setIsFormOpen] = useState(false);
  const [isDeleteAlertOpen, setIsDeleteAlertOpen] = useState(false);
  const [selectedQuestion, setSelectedQuestion] = useState<Question | null>(null);
  const { toast } = useToast();

  const handleSave = (data: QuestionFormData) => {
    if (selectedQuestion) {
      updateQuestion({ ...selectedQuestion, ...data });
      toast({ title: "Question Updated", description: "The question has been successfully updated." });
    } else {
      addQuestion(data);
      toast({ title: "Question Added", description: "The new question has been added to the quiz." });
    }
    setSelectedQuestion(null);
  };
  
  const openDeleteDialog = (question: Question) => {
    setSelectedQuestion(question);
    setIsDeleteAlertOpen(true);
  }

  const handleDelete = () => {
    if (selectedQuestion) {
        removeQuestion(selectedQuestion.id);
        toast({ title: "Question Deleted", description: "The question has been removed.", variant: "destructive" });
        setSelectedQuestion(null);
    }
  }

  const openForm = (question: Question | null) => {
    setSelectedQuestion(question);
    setIsFormOpen(true);
  }

  if (!isInitialized) {
    return <div className="flex justify-center items-center h-64"><BrainCircuit className="h-12 w-12 animate-pulse text-primary" /></div>;
  }

  return (
    <div className="space-y-4">
      <div className="flex justify-end">
        <Dialog open={isFormOpen} onOpenChange={setIsFormOpen}>
          <DialogTrigger asChild>
            <Button onClick={() => openForm(null)}>
              <PlusCircle className="mr-2 h-4 w-4" />
              Add Question
            </Button>
          </DialogTrigger>
          <DialogContent className="sm:max-w-[625px]">
            <DialogHeader>
              <DialogTitle className="font-headline">{selectedQuestion ? 'Edit Question' : 'Add New Question'}</DialogTitle>
            </DialogHeader>
            <QuestionForm
              question={selectedQuestion}
              onSave={handleSave}
              onClose={() => setIsFormOpen(false)}
            />
          </DialogContent>
        </Dialog>
      </div>

      <Card>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Question</TableHead>
              <TableHead className="w-[150px]">Category</TableHead>
              <TableHead className="w-[100px] text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {questions.length > 0 ? (
              questions.map((q) => (
                <TableRow key={q.id}>
                  <TableCell className="font-medium truncate max-w-sm">{q.questionText}</TableCell>
                  <TableCell>{q.category}</TableCell>
                  <TableCell className="text-right">
                    <div className="flex items-center justify-end gap-2">
                      <Button variant="ghost" size="icon" onClick={() => openForm(q)}>
                        <Edit className="h-4 w-4" />
                        <span className="sr-only">Edit</span>
                      </Button>
                      <Button variant="ghost" size="icon" className="text-destructive hover:text-destructive" onClick={() => openDeleteDialog(q)}>
                        <Trash2 className="h-4 w-4" />
                        <span className="sr-only">Delete</span>
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={3} className="text-center h-24">No questions found.</TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </Card>
      
      <AlertDialog open={isDeleteAlertOpen} onOpenChange={setIsDeleteAlertOpen}>
        <AlertDialogContent>
            <AlertDialogHeader>
            <AlertDialogTitle>Are you sure?</AlertDialogTitle>
            <AlertDialogDescription>
                This action cannot be undone. This will permanently delete the question.
            </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
            <AlertDialogCancel onClick={() => setSelectedQuestion(null)}>Cancel</AlertDialogCancel>
            <AlertDialogAction onClick={handleDelete}>Delete</AlertDialogAction>
            </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
